'use client';

import { useState, useEffect } from 'react';
import { motion } from 'framer-motion';
import { CheckCircle2, Sparkles, Award, Shield, Zap, Target, TrendingUp, Globe2 } from 'lucide-react';
import { useLanguage } from '@/contexts/language-context';

export default function WhyChooseUs() {
  const { content } = useLanguage();
  const { whyChooseUs } = content;
  const [mousePosition, setMousePosition] = useState({ x: 0, y: 0 });
  const [particles, setParticles] = useState([]);

  useEffect(() => {
    const generateParticles = () => {
      if (typeof window === 'undefined') return [];
      return [...Array(20)].map((_, i) => ({
        id: i,
        x: Math.random() * 100,
        y: Math.random() * 100,
        size: Math.random() * 4 + 2,
        duration: Math.random() * 3 + 2,
        delay: Math.random() * 2
      }));
    };
    setParticles(generateParticles());
  }, []);

  const handleMouseMove = (e) => {
    if (typeof window === 'undefined') return;
    const rect = e.currentTarget.getBoundingClientRect();
    setMousePosition({
      x: ((e.clientX - rect.left) / rect.width) * 100,
      y: ((e.clientY - rect.top) / rect.height) * 100
    });
  };

  const features = [
    { 
      icon: Award, 
      label: whyChooseUs.features.excellence,
      color: 'from-amber-400 to-yellow-500',
      bg: 'bg-amber-500/20',
      glow: 'rgba(251, 191, 36, 0.5)'
    },
    { 
      icon: Shield, 
      label: whyChooseUs.features.reliability,
      color: 'from-blue-400 to-blue-600',
      bg: 'bg-blue-500/20',
      glow: 'rgba(59, 130, 246, 0.5)'
    },
    { 
      icon: Sparkles, 
      label: whyChooseUs.features.innovation,
      color: 'from-purple-400 to-pink-500',
      bg: 'bg-purple-500/20',
      glow: 'rgba(168, 85, 247, 0.5)'
    },
    { 
      icon: Target, 
      label: whyChooseUs.features.precision,
      color: 'from-emerald-400 to-green-500',
      bg: 'bg-emerald-500/20',
      glow: 'rgba(16, 185, 129, 0.5)'
    },
  ];

  const stats = [
    { value: '250+', label: whyChooseUs.stats.projects, icon: TrendingUp },
    { value: '15+', label: whyChooseUs.stats.experience, icon: Award },
    { value: '50+', label: whyChooseUs.stats.partners, icon: Globe2 },
  ];

  return (
    <section id="why-us" className="scroll-mt-20 py-20 px-4 relative overflow-hidden bg-gradient-to-br from-stone-50 via-amber-50/30 to-slate-50">
      {/* Background Decorative Elements */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute top-0 left-0 w-full h-full opacity-[0.02]">
          <div className="absolute inset-0" style={{
            backgroundImage: `radial-gradient(circle at 50% 50%, rgba(0,0,0,0.1) 1px, transparent 1px)`,
            backgroundSize: '40px 40px'
          }} />
        </div>
      </div>

      <div className="relative max-w-7xl mx-auto">
        {/* Main Card */}
        <div 
          className="relative overflow-hidden rounded-[2rem] shadow-2xl"
          onMouseMove={handleMouseMove}
        >
          {/* Background Image Overlay */}
          <div className="absolute inset-0">
            <img
              src="https://images.unsplash.com/photo-1504328345606-18bbc8c9d7d1?w=1920&q=80"
              alt="Industrial Excellence"
              className="w-full h-full object-cover"
            />
            {/* Multi-layer Gradient Overlay */}
            <div className="absolute inset-0 bg-gradient-to-br from-blue-950/95 via-slate-900/90 to-blue-900/95" />
            <div className="absolute inset-0 bg-gradient-to-t from-black/50 via-transparent to-transparent" />
            <div className="absolute inset-0 bg-[radial-gradient(circle_at_50%_50%,rgba(251,191,36,0.1),transparent_70%)]" />
          </div>

          {/* Animated Mesh Pattern */}
          <div className="absolute inset-0 opacity-10">
            <svg className="w-full h-full" xmlns="http://www.w3.org/2000/svg">
              <defs>
                <pattern id="mesh" x="0" y="0" width="60" height="60" patternUnits="userSpaceOnUse">
                  <circle cx="30" cy="30" r="1" fill="white" opacity="0.5" />
                  <circle cx="0" cy="0" r="1" fill="white" opacity="0.3" />
                  <circle cx="60" cy="0" r="1" fill="white" opacity="0.3" />
                  <circle cx="0" cy="60" r="1" fill="white" opacity="0.3" />
                  <circle cx="60" cy="60" r="1" fill="white" opacity="0.3" />
                </pattern>
              </defs>
              <rect width="100%" height="100%" fill="url(#mesh)">
                <animateTransform
                  attributeName="transform"
                  type="translate"
                  from="0 0"
                  to="60 60"
                  dur="20s"
                  repeatCount="indefinite"
                />
              </rect>
            </svg>
          </div>

          {/* Floating Orbs */}
          <div className="absolute inset-0 overflow-hidden pointer-events-none">
            <div 
              className="absolute w-96 h-96 rounded-full bg-gradient-to-br from-amber-500/20 to-yellow-600/10 blur-[100px] animate-pulse"
              style={{ 
                top: '10%', 
                left: '10%',
                animationDuration: '4s'
              }}
            />
            <div 
              className="absolute w-80 h-80 rounded-full bg-gradient-to-br from-blue-500/20 to-cyan-600/10 blur-[90px] animate-pulse"
              style={{ 
                bottom: '15%', 
                right: '15%',
                animationDuration: '5s',
                animationDelay: '1s'
              }}
            />
            <div 
              className="absolute w-72 h-72 rounded-full bg-gradient-to-br from-emerald-500/15 to-green-600/10 blur-[80px] animate-pulse"
              style={{ 
                top: '50%', 
                right: '20%',
                animationDuration: '6s',
                animationDelay: '2s'
              }}
            />
          </div>

          {/* Floating Particles */}
          <div className="absolute inset-0 overflow-hidden pointer-events-none">
            {particles.map((particle) => (
              <div
                key={particle.id}
                className="absolute rounded-full bg-white/40"
                style={{
                  left: `${particle.x}%`,
                  top: `${particle.y}%`,
                  width: `${particle.size}px`,
                  height: `${particle.size}px`,
                  animation: `float ${particle.duration}s ease-in-out infinite`,
                  animationDelay: `${particle.delay}s`,
                  boxShadow: '0 0 10px rgba(255,255,255,0.5)'
                }}
              />
            ))}
          </div>

          {/* Mouse Spotlight Effect */}
          <div 
            className="absolute w-[500px] h-[500px] rounded-full bg-amber-400/10 blur-[100px] pointer-events-none opacity-0 hover:opacity-100 transition-opacity duration-700"
            style={{
              left: `${mousePosition.x}%`,
              top: `${mousePosition.y}%`,
              transform: 'translate(-50%, -50%)'
            }}
          />

          {/* Content */}
          <div className="relative z-10 px-8 py-20 sm:px-12 lg:px-20">
            <div className="mx-auto max-w-5xl">
              {/* Icon Section */}
              <div className="text-center mb-12">
                <div className="relative inline-flex mb-8">
                  {/* Rotating Ring */}
                  <div className="absolute inset-0 rounded-full animate-spin-slow">
                    <div className="w-full h-full rounded-full border-2 border-dashed border-amber-400/30" />
                  </div>
                  
                  {/* Pulsing Glow */}
                  <div className="absolute inset-0 rounded-full bg-amber-400/20 blur-2xl animate-pulse" />
                  
                  {/* Icon Container */}
                  <div className="relative rounded-full bg-gradient-to-br from-amber-500/30 to-yellow-600/20 p-6 backdrop-blur-sm border-2 border-amber-400/40 shadow-2xl">
                    <CheckCircle2 className="h-16 w-16 text-amber-400 drop-shadow-[0_0_20px_rgba(251,191,36,0.8)]" />
                  </div>
                </div>

                {/* Badge */}
                <div className="inline-flex items-center gap-2 px-5 py-2 bg-white/10 backdrop-blur-xl rounded-full border border-white/20 mb-6">
                  <span className="text-sm font-bold text-amber-400 tracking-wide">
                    {whyChooseUs.badge}
                  </span>
                </div>

                {/* Title */}
                <h2 className="font-bold text-5xl sm:text-6xl lg:text-7xl mb-8 leading-tight">
                  <span className="block bg-gradient-to-r from-white via-blue-100 to-white bg-clip-text text-transparent drop-shadow-2xl">
                    {whyChooseUs.title}
                  </span>
                </h2>

                {/* Decorative Line */}
                <div className="flex items-center justify-center gap-4 mb-8">
                  <div className="h-[2px] w-20 bg-gradient-to-r from-transparent via-amber-400 to-amber-400 rounded-full" />
                  <div className="relative">
                    <div className="w-3 h-3 rounded-full bg-gradient-to-r from-amber-400 to-yellow-500 shadow-lg animate-pulse" />
                    <div className="absolute inset-0 bg-amber-400/50 rounded-full blur-md" />
                  </div>
                  <div className="h-[2px] w-20 bg-gradient-to-l from-transparent via-amber-400 to-amber-400 rounded-full" />
                </div>

                {/* Description */}
                <p className="text-xl sm:text-2xl leading-relaxed text-slate-200 max-w-3xl mx-auto drop-shadow-lg">
                  {whyChooseUs.text}
                </p>
              </div>

              {/* Features Grid */}
              <div className="grid grid-cols-2 lg:grid-cols-4 gap-6 mb-16">
                {features.map((feature, index) => (
                  <div
                    key={index}
                    className="group relative"
                    style={{
                      animation: `fadeInUp 0.6s ease-out ${index * 0.1}s both`
                    }}
                  >
                    {/* Glow Effect */}
                    <div className={`absolute -inset-1 bg-gradient-to-r ${feature.color} rounded-2xl blur-lg opacity-0 group-hover:opacity-50 transition-opacity duration-500`} />
                    
                    {/* Card */}
                    <div className={`relative ${feature.bg} backdrop-blur-xl rounded-2xl p-6 border-2 border-white/20 group-hover:border-white/40 transition-all duration-500 group-hover:scale-105 shadow-xl`}>
                      <div className="flex flex-col items-center gap-4">
                        {/* Icon */}
                        <div className={`relative p-4 rounded-xl bg-gradient-to-br ${feature.color} shadow-lg group-hover:scale-110 transition-transform duration-500`}>
                          <feature.icon className="w-8 h-8 text-white" />
                          <div className="absolute inset-0 bg-white/20 rounded-xl blur-lg opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
                        </div>
                        
                        {/* Label */}
                        <span className="text-sm font-bold text-white tracking-wide text-center">
                          {feature.label}
                        </span>
                      </div>

                      {/* Floating Particle on Hover */}
                      <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 opacity-0 group-hover:opacity-100 pointer-events-none">
                        <div className={`w-2 h-2 rounded-full bg-gradient-to-r ${feature.color}`} 
                             style={{ 
                               animation: 'floatParticle 2s ease-out infinite',
                               boxShadow: `0 0 20px ${feature.glow}`
                             }} 
                        />
                      </div>
                    </div>
                  </div>
                ))}
              </div>

              {/* Stats Section */}
              <div className="grid grid-cols-1 sm:grid-cols-3 gap-6">
                {stats.map((stat, index) => (
                  <div
                    key={index}
                    className="group relative bg-white/5 backdrop-blur-xl rounded-2xl p-6 border border-white/10 hover:border-amber-400/40 transition-all duration-500 hover:bg-white/10"
                    style={{
                      animation: `fadeInUp 0.6s ease-out ${0.4 + index * 0.1}s both`
                    }}
                  >
                    <div className="flex flex-col items-center gap-3">
                      <div className="p-3 rounded-lg bg-gradient-to-br from-amber-500/20 to-yellow-600/20 group-hover:scale-110 transition-transform duration-300">
                        <stat.icon className="w-6 h-6 text-amber-400" />
                      </div>
                      <div className="text-center">
                        <div className="text-3xl font-bold bg-gradient-to-r from-amber-400 to-yellow-500 bg-clip-text text-transparent mb-1">
                          {stat.value}
                        </div>
                        <div className="text-xs text-slate-300 font-medium">
                          {stat.label}
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>

              {/* Bottom Accent */}
              <div className="mt-16 flex justify-center">
                <div className="h-1 w-32 rounded-full bg-gradient-to-r from-transparent via-amber-400 to-transparent shadow-lg shadow-amber-400/50" />
              </div>
            </div>
          </div>

          {/* Corner Decorations */}
          <div className="absolute top-0 left-0 w-40 h-40 border-t-2 border-l-2 border-amber-400/20 rounded-tl-3xl" />
          <div className="absolute bottom-0 right-0 w-40 h-40 border-b-2 border-r-2 border-amber-400/20 rounded-br-3xl" />

          {/* Bottom Shadow */}
          <div className="absolute bottom-0 left-0 right-0 h-32 bg-gradient-to-t from-black/40 to-transparent pointer-events-none" />
        </div>
      </div>

      {/* CSS Animations */}
      <style jsx>{`
        @keyframes float {
          0%, 100% { 
            transform: translateY(0px) translateX(0px);
            opacity: 0.3;
          }
          25% { 
            transform: translateY(-20px) translateX(10px);
            opacity: 0.6;
          }
          50% { 
            transform: translateY(-40px) translateX(-10px);
            opacity: 1;
          }
          75% { 
            transform: translateY(-20px) translateX(5px);
            opacity: 0.6;
          }
        }

        @keyframes floatParticle {
          0% {
            transform: translate(-50%, -50%) translateY(0);
            opacity: 0;
          }
          50% {
            opacity: 1;
          }
          100% {
            transform: translate(-50%, -50%) translateY(-30px);
            opacity: 0;
          }
        }

        @keyframes fadeInUp {
          from {
            opacity: 0;
            transform: translateY(20px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }

        @keyframes spin-slow {
          from { transform: rotate(0deg); }
          to { transform: rotate(360deg); }
        }

        .animate-spin-slow {
          animation: spin-slow 10s linear infinite;
        }
      `}</style>
    </section>
  );
}

    